/*
 * Decompiled with CFR 0.152.
 */
package de.pontonconsulting.xmlpipe.security.certificate;

import de.pontonconsulting.xmlpipe.security.CertificateReader;
import de.pontonconsulting.xmlpipe.security.CertificateUtility;
import de.pontonconsulting.xmlpipe.security.certificate.ICertificateFilter;
import de.pontonconsulting.xmlpipe.security.certificate.SortCriteria;
import java.security.cert.CertificateException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import javax.security.auth.x500.X500Principal;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class CertificateChooser {
    private static final Logger LOGGER = LogManager.getLogger((String)"Messenger.CertificateChooser");
    private static final long ONE_YEAR = 1314000000L;
    private final CertificateReader certificateInfo;
    private final CertificateUtility certificateUtility;

    public CertificateChooser(CertificateReader certificateInfo, CertificateUtility certificateUtility) {
        Objects.requireNonNull(certificateInfo, "CertificateInfo can't be null.");
        Objects.requireNonNull(certificateUtility, "CertificateUtility can't be null.");
        this.certificateInfo = certificateInfo;
        this.certificateUtility = certificateUtility;
    }

    public X509Certificate getNextMatchingCertificate(List<ICertificateFilter> certificateFilters, List<SortCriteria> sortCriteria, X509Certificate currentCertificate, Collection<X509Certificate> certificates) {
        List<X509Certificate> filteredCertificates = new ArrayList<X509Certificate>(certificates);
        boolean removedCurrentCertificate = filteredCertificates.remove(currentCertificate);
        for (ICertificateFilter certificateFilter : certificateFilters) {
            filteredCertificates = certificateFilter.doFilter(currentCertificate, filteredCertificates);
        }
        List<CertificateWithPoints> certificateWithPointsList = this.getCertificateWithPointsList(sortCriteria, currentCertificate, filteredCertificates);
        Collections.sort(certificateWithPointsList);
        if (certificateWithPointsList.isEmpty()) {
            if (removedCurrentCertificate) {
                return currentCertificate;
            }
            return null;
        }
        X509Certificate nextCertificate = certificateWithPointsList.getLast().getCertificate();
        return currentCertificate.getNotAfter().compareTo(nextCertificate.getNotAfter()) > 0 ? currentCertificate : nextCertificate;
    }

    private List<CertificateWithPoints> getCertificateWithPointsList(List<SortCriteria> sortCriteria, X509Certificate currentCertificate, List<X509Certificate> certificates) {
        ArrayList<CertificateWithPoints> certificateWithPointsList = new ArrayList<CertificateWithPoints>();
        for (X509Certificate certificate : certificates) {
            certificateWithPointsList.add(new CertificateWithPoints(this, certificate, this.computePoints(sortCriteria, currentCertificate, certificate)));
        }
        return certificateWithPointsList;
    }

    private int computePoints(List<SortCriteria> sortCriteria, X509Certificate currentCertificate, X509Certificate anotherCertificate) {
        int points = 0;
        LOGGER.debug("starting to compute points for certificate{}", (Object)this.certificateUtility.toUsingInfoText(anotherCertificate, ""));
        for (SortCriteria criteria : sortCriteria) {
            switch (criteria) {
                case SUBJECT: {
                    if (!this.subjectEqual(currentCertificate, anotherCertificate)) break;
                    LOGGER.debug("found equal subject for certificate{}", (Object)this.certificateUtility.toUsingInfoText(anotherCertificate, ""));
                    points += criteria.getPoints();
                    break;
                }
                case CN: {
                    if (!this.cnEqual(currentCertificate, anotherCertificate)) break;
                    LOGGER.debug("found equal CN for certificate{}", (Object)this.certificateUtility.toUsingInfoText(anotherCertificate, ""));
                    points += criteria.getPoints();
                    break;
                }
                case ISSUER: {
                    if (!this.issuerEqual(currentCertificate, anotherCertificate)) break;
                    LOGGER.debug("found equal issuer for certificate{}", (Object)this.certificateUtility.toUsingInfoText(anotherCertificate, ""));
                    points += criteria.getPoints();
                    break;
                }
                case KEY_USAGE: {
                    if (!this.keyUsageEqual(currentCertificate, anotherCertificate)) break;
                    LOGGER.debug("found equal key usage for certificate{}", (Object)this.certificateUtility.toUsingInfoText(anotherCertificate, ""));
                    points += criteria.getPoints();
                    break;
                }
                case EXTENDED_KEY_USAGE: {
                    if (!this.extendedKeyUsageEqual(currentCertificate, anotherCertificate)) break;
                    LOGGER.debug("found equal extended key usage for certificate{}", (Object)this.certificateUtility.toUsingInfoText(anotherCertificate, ""));
                    points += criteria.getPoints();
                }
            }
        }
        long difference = anotherCertificate.getNotAfter().getTime() - currentCertificate.getNotAfter().getTime();
        int validatyPoints = (int)(difference / 1314000000L);
        LOGGER.debug("computed {} points for certificate{}", (Object)(points += validatyPoints), (Object)this.certificateUtility.toUsingInfoText(anotherCertificate, ""));
        return points;
    }

    private boolean extendedKeyUsageEqual(X509Certificate currentCertificate, X509Certificate anotherCertificate) {
        try {
            List<String> extendedKeyUsageOfCurrentCertificate = currentCertificate.getExtendedKeyUsage();
            List<String> extendedKeyUsageOfNewCertificate = anotherCertificate.getExtendedKeyUsage();
            if (extendedKeyUsageOfCurrentCertificate == extendedKeyUsageOfNewCertificate) {
                return true;
            }
            if (extendedKeyUsageOfCurrentCertificate == null || extendedKeyUsageOfNewCertificate == null) {
                return false;
            }
            ArrayList<String> currentList = new ArrayList<String>(extendedKeyUsageOfCurrentCertificate);
            ArrayList<String> newList = new ArrayList<String>(extendedKeyUsageOfNewCertificate);
            Collections.sort(currentList);
            Collections.sort(newList);
            return currentList.equals(newList);
        }
        catch (CertificateParsingException e) {
            LOGGER.error("Could not read extended key usage.", (Throwable)e);
            return false;
        }
    }

    private boolean keyUsageEqual(X509Certificate currentCertificate, X509Certificate anotherCertificate) {
        return Arrays.equals(currentCertificate.getKeyUsage(), anotherCertificate.getKeyUsage());
    }

    private boolean subjectEqual(X509Certificate currentCertificate, X509Certificate anotherCertificate) {
        X500Principal subjectPrincipal = currentCertificate.getSubjectX500Principal();
        return subjectPrincipal.equals(anotherCertificate.getSubjectX500Principal());
    }

    private boolean cnEqual(X509Certificate currentCertificate, X509Certificate anotherCertificate) {
        try {
            String cn = this.certificateInfo.retrieveCN(currentCertificate);
            return cn.equalsIgnoreCase(this.certificateInfo.retrieveCN(anotherCertificate));
        }
        catch (CertificateException certificateException) {
            return false;
        }
    }

    private boolean issuerEqual(X509Certificate currentCertificate, X509Certificate anotherCrtificate) {
        X500Principal issuerPrincipal = currentCertificate.getIssuerX500Principal();
        return issuerPrincipal.equals(anotherCrtificate.getIssuerX500Principal());
    }

    private class CertificateWithPoints
    implements Comparable<CertificateWithPoints> {
        private final X509Certificate certificate;
        private final int points;

        private CertificateWithPoints(CertificateChooser certificateChooser, X509Certificate certificate, int points) {
            this.certificate = certificate;
            this.points = points;
        }

        X509Certificate getCertificate() {
            return this.certificate;
        }

        @Override
        public int compareTo(CertificateWithPoints anotherCertificateWithPoints) {
            int compareToResult = this.points - anotherCertificateWithPoints.points;
            return compareToResult == 0 ? this.certificate.getNotAfter().compareTo(anotherCertificateWithPoints.certificate.getNotAfter()) : compareToResult;
        }
    }
}

