/*
 * Decompiled with CFR 0.152.
 */
package de.ponton.application.update.applicationpart;

import de.ponton.application.update.api.SystemOutLogger;
import de.ponton.application.update.api.communication.CommunicationData;
import de.ponton.application.update.api.content.IDiffEntry;
import de.ponton.application.update.api.patch.PatchReader;
import de.ponton.application.update.api.wrapperconf.OsType;
import de.ponton.application.update.api.wrapperconf.WrapperConf;
import de.ponton.application.update.applicationpart.exception.PatchException;
import de.ponton.application.update.applicationpart.io.DeleteFileVisitor;
import de.ponton.application.update.applicationpart.io.PermissionChecker;
import de.ponton.application.update.applicationpart.patchinfo.PatchInfo;
import de.ponton.application.update.applicationpart.patchinfo.PatchInfoBuilder;
import de.ponton.application.update.applicationpart.patchinfo.Version;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.cert.X509Certificate;
import java.text.ParseException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.SortedSet;
import shaded.org.apache.commons.configuration2.ex.ConfigurationException;
import shaded.org.apache.commons.lang3.StringUtils;
import shaded.org.apache.commons.lang3.SystemUtils;

public class PatchProcessor {
    private static final SystemOutLogger LOG = new SystemOutLogger(PatchReader.class);
    private static final Path APP_HOME_PATH = new File(".").toPath();
    private static final String PATCH_WRAPPER_CONF_FILENAME = "wrapper.conf_patch";
    private final String appID;
    private final Version currentAppVersion;
    private final Version currentJavaVersion;
    private final OsType currentOsType;
    private final List<X509Certificate> certList;
    private final File updateFolder;
    private final String patchFilename;
    private final PermissionChecker permissionChecker;
    private final Version currentApiVersion;
    private final Version currentConfigVersion;
    private final Map<String, String> installedAddOns;
    private final String wrapperConfFolder;

    public PatchProcessor(File updateFolder, String patchFilename, List<X509Certificate> certList, String appID, String currentAppVersionString) throws ParseException, PatchException {
        this(updateFolder, patchFilename, certList, appID, currentAppVersionString, "1.0", currentAppVersionString, new HashMap<String, String>(0));
    }

    public PatchProcessor(File updateFolder, String patchFilename, List<X509Certificate> certList, String appID, String currentAppVersionString, String currentApiVersionString, String currentConfigVersionString, Map<String, String> installedAddOns) throws ParseException, PatchException {
        this(updateFolder, patchFilename, certList, appID, currentAppVersionString, currentApiVersionString, currentConfigVersionString, installedAddOns, "launcher/conf/");
    }

    public PatchProcessor(File updateFolder, String patchFilename, List<X509Certificate> certList, String appID, String currentAppVersionString, String currentApiVersionString, String currentConfigVersionString, Map<String, String> installedAddOns, String wrapperConfFolder) throws ParseException, PatchException {
        Objects.requireNonNull(updateFolder, "Update folder is required.");
        Objects.requireNonNull(patchFilename, "Patch filename is required.");
        Objects.requireNonNull(certList, "Trusted root CA certificates are required.");
        Objects.requireNonNull(appID, "AppID is required.");
        Objects.requireNonNull(currentAppVersionString, "Current application version is required.");
        Objects.requireNonNull(currentApiVersionString, "Current api version is required.");
        Objects.requireNonNull(currentConfigVersionString, "Current config version is required.");
        Objects.requireNonNull(installedAddOns, "Map of installed addOns can't be NULL.");
        this.updateFolder = updateFolder;
        this.patchFilename = patchFilename;
        this.certList = certList;
        this.appID = appID;
        this.currentAppVersion = new Version(currentAppVersionString);
        this.currentJavaVersion = this.determineCurrentJavaVersion();
        this.currentOsType = this.determineCurrentOSType();
        this.permissionChecker = new PermissionChecker(APP_HOME_PATH);
        this.currentApiVersion = new Version(currentApiVersionString);
        this.currentConfigVersion = new Version(currentConfigVersionString);
        this.installedAddOns = installedAddOns;
        this.wrapperConfFolder = wrapperConfFolder;
    }

    private Version determineCurrentJavaVersion() throws ParseException {
        String javaVersionString = System.getProperty("java.runtime.version");
        return new Version(javaVersionString);
    }

    private OsType determineCurrentOSType() throws PatchException {
        String osArch = SystemUtils.OS_ARCH;
        boolean isX64 = osArch != null ? osArch.contains("64") : true;
        if (SystemUtils.IS_OS_WINDOWS) {
            return isX64 ? OsType.WINDOWS_x64 : OsType.WINDOWS_x86;
        }
        if (SystemUtils.IS_OS_LINUX) {
            return isX64 ? OsType.LINUX_x64 : OsType.LINUX_x86;
        }
        if (SystemUtils.IS_OS_MAC_OSX) {
            return OsType.MAC_OS;
        }
        throw new PatchException(String.format("The application is running on not supported OS (architecture: %s; name: %s; version: %s).", SystemUtils.OS_ARCH, SystemUtils.OS_NAME, SystemUtils.OS_VERSION));
    }

    public void execute() throws PatchException {
        if (this.purgeOldPatch()) {
            return;
        }
        try {
            File patchFile = new File(this.updateFolder, this.patchFilename);
            LOG.logInfo("Looking for patch file: " + patchFile.getAbsolutePath());
            if (patchFile.exists()) {
                LOG.logInfo("Processing Patch[" + patchFile.getName() + "]");
                PatchReader reader = new PatchReader(patchFile, this.certList);
                try {
                    reader.verifySignedJar();
                    WrapperConf patchWrapperConf = reader.getWrapperConf();
                    PatchInfo patchInfo = PatchInfoBuilder.buildPatchInfo(patchFile, patchWrapperConf);
                    this.checkPatchPreconditions(patchInfo);
                    LOG.logInfo("Found valid patch file. [" + patchFile.getPath() + "]");
                    this.checkReadWritePermission(reader);
                    File wrapperConfFile = this.getAppWrapperConfFile();
                    this.switchWrapperConf(wrapperConfFile, patchWrapperConf);
                    System.exit(100);
                    return;
                }
                catch (PatchException | IOException e) {
                    LOG.logError("There was an error processing Patch[" + patchFile.getName() + "]: " + e.getMessage());
                }
            }
            LOG.logInfo("No executable patch found");
        }
        catch (Exception e) {
            throw new PatchException("There was an error processing the patches: " + e.getMessage(), e);
        }
    }

    void checkPatchPreconditions(PatchInfo patchToCheck) throws IOException, ConfigurationException, PatchException {
        this.checkPatchJavaMinVersion(patchToCheck);
        this.checkPatchForAppID(patchToCheck);
        this.checkPatchForAppVersion(patchToCheck);
        this.checkPatchValidForOS(patchToCheck);
        if (StringUtils.isNotBlank(patchToCheck.getAddonID())) {
            this.checkAddonApiVersion(patchToCheck);
            this.checkPostPatchAddonVersion(patchToCheck);
            this.checkAddonConfigVersion(patchToCheck);
        } else {
            this.checkAfterPatchAppVersion(patchToCheck);
            this.checkAppJavaMinVersion(patchToCheck);
            this.checkPostPatchAppJavaVersion(patchToCheck);
        }
    }

    private void checkAddonConfigVersion(PatchInfo patchToCheck) throws PatchException {
        if (patchToCheck.getAddonConfigVersion() != null && this.currentConfigVersion.compareTo(patchToCheck.getAddonConfigVersion()) != 0) {
            throw new PatchException(String.format("The patch has unsupported config '%s'.", patchToCheck.getAddonConfigVersion()));
        }
    }

    private void checkAddonApiVersion(PatchInfo patchToCheck) throws PatchException {
        if (patchToCheck.getAddonApiVersion() == null || this.currentApiVersion.compareTo(patchToCheck.getAddonApiVersion()) != 0) {
            throw new PatchException(String.format("The patch has unsupported API '%s'.", patchToCheck.getAddonApiVersion()));
        }
    }

    private void checkPostPatchAddonVersion(PatchInfo patchToCheck) throws PatchException {
        String currentAddOnVersionString = this.installedAddOns.get(patchToCheck.getAddonID());
        if (StringUtils.isNoneBlank(currentAddOnVersionString)) {
            try {
                Version currentAddOnVersion = new Version(currentAddOnVersionString);
                if (patchToCheck.getPostPatchAddonVersion() == null) {
                    throw new PatchException("No addon version set in the patch.");
                }
                if (currentAddOnVersion.compareTo(patchToCheck.getPostPatchAddonVersion()) >= 0) {
                    if (currentAddOnVersion.compareTo(patchToCheck.getPostPatchAddonVersion()) == 0) {
                        throw new PatchException(String.format("The already installed addon version is the same '%s'.", patchToCheck.getPostPatchAddonVersion()));
                    }
                    throw new PatchException(String.format("The already installed addon version '%s' is newer than '%s'.", currentAddOnVersion, patchToCheck.getPostPatchAddonVersion()));
                }
            }
            catch (ParseException e) {
                throw new PatchException(String.format("Could not read version of installed addon '%s'", patchToCheck.getAddonID()), e);
            }
        }
    }

    private void checkPostPatchAppJavaVersion(PatchInfo patchToCheck) throws PatchException, IOException, ConfigurationException {
        if (patchToCheck.getPostPatchAppJavaVersion() != null) {
            WrapperConf appWrapperConf = this.getAppWrapperConf();
            String oldJavaPath = appWrapperConf.getProperty("wrapper.java.command");
            if (StringUtils.isBlank(oldJavaPath)) {
                throw new PatchException("No java path defined in the application.");
            }
            if (oldJavaPath.equals(patchToCheck.getNewJavaPath()) && !this.currentJavaVersion.equals(patchToCheck.getPostPatchAppJavaVersion())) {
                throw new PatchException(String.format("The defined new java path in the patch '%s' is already in use with java version '%s', but '%s' is set in the patch.", patchToCheck.getNewJavaPath(), this.currentJavaVersion, patchToCheck.getPostPatchAppJavaVersion()));
            }
        }
    }

    private void checkAppJavaMinVersion(PatchInfo patchToCheck) throws PatchException {
        if (patchToCheck.getMinAppJavaVersion() == null || this.currentJavaVersion.compareTo(patchToCheck.getMinAppJavaVersion()) < 0) {
            throw new PatchException(String.format("The new application version needs at least Java '%s'.", patchToCheck.getMinAppJavaVersion()));
        }
    }

    private void checkAfterPatchAppVersion(PatchInfo patchToCheck) throws PatchException {
        if (patchToCheck.getPostPatchAppVersion() == null) {
            throw new PatchException("No new application version is defined in the patch.");
        }
        if (this.currentAppVersion.compareTo(patchToCheck.getPostPatchAppVersion()) > 0) {
            throw new PatchException(String.format("The current application version '%s' is newer than '%s'.", this.currentAppVersion, patchToCheck.getPostPatchAppVersion()));
        }
        if (this.currentAppVersion.compareTo(patchToCheck.getPostPatchAppVersion()) == 0) {
            throw new PatchException(String.format("The current application version '%s' is already installed.", patchToCheck.getPostPatchAppVersion()));
        }
    }

    private void checkPatchForAppID(PatchInfo patchToCheck) throws PatchException {
        if (!StringUtils.isNotBlank(patchToCheck.getAppID()) || !this.appID.equals(patchToCheck.getAppID())) {
            throw new PatchException(String.format("The patch is for %s only.", patchToCheck.getAppID()));
        }
    }

    private void checkPatchForAppVersion(PatchInfo patchToCheck) throws PatchException {
        if (patchToCheck.getMinAppVersion() == null || this.currentAppVersion.compareTo(patchToCheck.getMinAppVersion()) < 0) {
            throw new PatchException(String.format("The patch needs at least application version '%s'.", patchToCheck.getMinAppVersion()));
        }
    }

    private void checkPatchJavaMinVersion(PatchInfo patchToCheck) throws PatchException {
        if (patchToCheck.getMinPatcherJavaVersion() == null || patchToCheck.getMinPatcherJavaVersion().compareTo(this.currentJavaVersion) > 0) {
            throw new PatchException(String.format("The patch needs at least Java version '%s'.", patchToCheck.getMinPatcherJavaVersion()));
        }
    }

    private void checkPatchValidForOS(PatchInfo patchToCheck) throws PatchException {
        if (!OsType.ALL.equals((Object)patchToCheck.getOsType()) && !this.currentOsType.equals((Object)patchToCheck.getOsType())) {
            throw new PatchException(String.format("The patch is only for OS type '%s'. ", new Object[]{patchToCheck.getOsType()}));
        }
    }

    private boolean purgeOldPatch() throws PatchException {
        File communicatioFile = new File(this.updateFolder, ".communication");
        if (communicatioFile.exists()) {
            CommunicationData communicationData = this.getCommunicationData(communicatioFile);
            this.logPatchResult(communicationData);
            this.removeOldJava(communicationData);
            this.removeFilesIn(this.updateFolder.toPath());
            return true;
        }
        return false;
    }

    private void removeOldJava(CommunicationData communicationData) throws PatchException {
        String oldJavaPath = communicationData.getOldJavaPath();
        if (StringUtils.isNotBlank(oldJavaPath)) {
            String appJavaPath;
            if (!new File(oldJavaPath).exists()) {
                LOG.logWarn("Old java folder doesn't exist: " + oldJavaPath);
                return;
            }
            try {
                WrapperConf appWrapperConf = this.getAppWrapperConf();
                appJavaPath = appWrapperConf.getProperty("wrapper.java.command");
            }
            catch (IOException | ConfigurationException e) {
                throw new PatchException(String.format("Could not read file 'wrapper.conf'.", new Object[0]), e);
            }
            if (oldJavaPath.trim().equals(appJavaPath.trim())) {
                LOG.logWarn("Using java folder is the same as the old used java folder. So the java folder will not be removed.");
            } else {
                Path javaHome = this.determineJavaHome(oldJavaPath);
                Path fullPath = APP_HOME_PATH.resolve(javaHome);
                LOG.logInfo("Removing old java path: " + fullPath);
                this.removeFilesIn(fullPath);
                try {
                    Files.delete(fullPath);
                }
                catch (IOException e) {
                    throw new PatchException(String.format("Could not remove root directory of old java.", new Object[0]), e);
                }
            }
        }
    }

    private Path determineJavaHome(String javaExecutablePath) {
        return Paths.get(javaExecutablePath, new String[0]).getParent().getParent();
    }

    private void logPatchResult(CommunicationData communicationData) {
        LOG.logInfo("Patch result code: " + (Object)((Object)communicationData.getResultCode()) + "; description: " + communicationData.getResultDescription());
    }

    private CommunicationData getCommunicationData(File communicatioFile) throws PatchException {
        try {
            return new CommunicationData(communicatioFile);
        }
        catch (IOException | ConfigurationException e) {
            throw new PatchException(String.format("Could not read communication file.", new Object[0]), e);
        }
    }

    private void removeFilesIn(Path path) throws PatchException {
        try {
            if (Files.exists(path, LinkOption.NOFOLLOW_LINKS)) {
                Files.walkFileTree(path, new DeleteFileVisitor(path));
            }
        }
        catch (IOException e) {
            throw new PatchException(String.format("Could not remove file/directory.", new Object[0]), e);
        }
    }

    void switchWrapperConf(File wrapperConfFile, WrapperConf patchWrapperConf) throws IOException, PatchException {
        File patchWrapperConfFile = new File(wrapperConfFile.getParentFile(), PATCH_WRAPPER_CONF_FILENAME);
        this.updateJVM(patchWrapperConf, wrapperConfFile);
        try {
            patchWrapperConf.writeTo(patchWrapperConfFile);
        }
        catch (IOException | ConfigurationException e) {
            throw new IOException("Could not write patch wrapper.conf file.", e);
        }
        File wrapperConfBackupFile = new File(wrapperConfFile.getParentFile(), "wrapper.conf_app");
        try {
            this.rename(wrapperConfFile, wrapperConfBackupFile);
        }
        catch (IOException e) {
            patchWrapperConfFile.delete();
        }
        try {
            this.rename(patchWrapperConfFile, wrapperConfFile);
        }
        catch (IOException e) {
            this.rename(wrapperConfBackupFile, wrapperConfFile);
            patchWrapperConfFile.delete();
        }
    }

    private void updateJVM(WrapperConf patchWrapperConf, File wrapperConfFile) throws IOException, PatchException {
        String jvmExecutablePath;
        try {
            WrapperConf appWrapperConf = new WrapperConf(wrapperConfFile);
            jvmExecutablePath = appWrapperConf.getProperty("wrapper.java.command");
        }
        catch (ConfigurationException e) {
            throw new PatchException(String.format("Could not read application wrapper.conf file.", new Object[0]), e);
        }
        patchWrapperConf.setProperty("wrapper.java.command", jvmExecutablePath);
    }

    private WrapperConf getAppWrapperConf() throws IOException, ConfigurationException {
        return new WrapperConf(this.getAppWrapperConfFile());
    }

    private File getAppWrapperConfFile() {
        return new File(this.wrapperConfFolder, "wrapper.conf");
    }

    private void rename(File sourceFile, File targetFile) throws IOException {
        if (!sourceFile.renameTo(targetFile)) {
            throw new IOException("Could not rename '" + sourceFile.getPath() + "' to '" + targetFile.getPath() + "'");
        }
    }

    private void checkReadWritePermission(PatchReader patchReader) throws PatchException, IOException {
        HashSet<Path> paths = new HashSet<Path>();
        paths.add(APP_HOME_PATH.resolve("launcher/conf/"));
        paths.add(this.updateFolder.toPath());
        SortedSet<IDiffEntry> allDiffEntries = patchReader.getAllDiffs();
        for (IDiffEntry diffEntry : allDiffEntries) {
            Path checkPath;
            Path rootPath = diffEntry.getPath().getRoot();
            if (rootPath == null || !Files.exists(checkPath = APP_HOME_PATH.resolve(rootPath), new LinkOption[0])) continue;
            paths.add(checkPath);
        }
        this.permissionChecker.checkPermissions(paths);
    }
}

