/*
 * Decompiled with CFR 0.152.
 */
package de.ponton.xp.adapter.api.internal;

import de.ponton.xp.adapter.api.ConnectionException;
import de.ponton.xp.adapter.api.NoConnectionAvailableException;
import de.ponton.xp.adapter.api.internal.OnCloseListener;
import de.ponton.xp.adapter.api.internal.PontonThreadFactory;
import de.ponton.xp.adapter.api.internal.WebSocketConnection;
import de.ponton.xp.adapter.api.internal.WebSocketConnectionFactory;
import java.net.URISyntaxException;
import java.net.http.WebSocket;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.logging.Logger;

public class WebSocketConnectionPool
implements OnCloseListener {
    private static final Logger LOG = Logger.getLogger(WebSocketConnectionPool.class.getName());
    private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1, new PontonThreadFactory("WebSocket-ConnectionPool-"));
    private final AtomicBoolean shuttingDown = new AtomicBoolean(false);
    private final ReentrantLock lock = new ReentrantLock();
    private final BlockingQueue<WebSocketConnection> availableWebSocketConnections = new LinkedBlockingQueue<WebSocketConnection>();
    private final Set<WebSocketConnection> borrowedWebSocketConnections = new HashSet<WebSocketConnection>();
    private final WebSocketConnectionFactory webSocketConnectionFactory;
    private final Map<WebSocket, WebSocketConnection> webSocketConnections = new ConcurrentHashMap<WebSocket, WebSocketConnection>();
    private Callable<Void> onReconnect;
    private final Consumer<Integer> totalConnectionCountConsumer;

    WebSocketConnectionPool(WebSocketConnectionFactory webSocketConnectionFactory, int parallelConnections, Consumer<Integer> totalConnectionCountConsumer) throws ConnectionException, URISyntaxException {
        this.webSocketConnectionFactory = webSocketConnectionFactory;
        this.totalConnectionCountConsumer = totalConnectionCountConsumer;
        for (int i = 0; i < parallelConnections; ++i) {
            this.addWebSocketConnection(webSocketConnectionFactory.create(this));
        }
    }

    @Override
    public void onClose(WebSocket webSocket) {
        this.lock.lock();
        try {
            WebSocketConnection webSocketConnection = this.webSocketConnections.get(webSocket);
            if (Objects.nonNull(webSocketConnection)) {
                this.removeSession(webSocketConnection);
                if (this.shuttingDown.get()) {
                    LOG.fine(() -> "ignore reconnect, because the adapter is shutting down.");
                } else {
                    LOG.fine(() -> "trying to reconnect loosed web socket connection.");
                    this.reconnectWebSocketConnection(0);
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void reconnectWebSocketConnection(int delay) {
        this.scheduler.schedule(() -> {
            try {
                this.addWebSocketConnection(this.webSocketConnectionFactory.create(this));
                if (this.onReconnect != null) {
                    this.onReconnect.call();
                }
            }
            catch (ConnectionException | URISyntaxException e) {
                LOG.severe(() -> "could not connect to messenger: " + String.valueOf(e));
                this.reconnectWebSocketConnection(5);
            }
            catch (Exception e) {
                LOG.severe(() -> "Unexpected error while reconnect: " + String.valueOf(e));
            }
        }, (long)delay, TimeUnit.SECONDS);
    }

    WebSocketConnection borrowConnection() throws NoConnectionAvailableException {
        this.lock.lock();
        try {
            WebSocketConnection connection = this.availableWebSocketConnections.poll(2L, TimeUnit.SECONDS);
            if (Objects.nonNull(connection) && !connection.getWebSocket().isInputClosed() && !connection.getWebSocket().isOutputClosed()) {
                this.borrowedWebSocketConnections.add(connection);
                WebSocketConnection webSocketConnection = connection;
                return webSocketConnection;
            }
            if (Objects.nonNull(connection)) {
                this.onClose(connection.getWebSocket());
                WebSocketConnection webSocketConnection = this.borrowConnection();
                return webSocketConnection;
            }
            try {
                throw new NoConnectionAvailableException("No connection available.");
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new NoConnectionAvailableException("No connection available: " + String.valueOf(e));
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    void releaseConnection(WebSocketConnection connection) {
        this.lock.lock();
        try {
            if (this.borrowedWebSocketConnections.remove(connection)) {
                this.availableWebSocketConnections.add(connection);
            }
        }
        finally {
            this.callOnConnectionChanged();
            this.lock.unlock();
        }
    }

    private void addWebSocketConnection(WebSocketConnection connection) {
        this.lock.lock();
        try {
            this.webSocketConnections.put(connection.getWebSocket(), connection);
            this.availableWebSocketConnections.add(connection);
        }
        finally {
            this.callOnConnectionChanged();
            this.lock.unlock();
        }
    }

    private void removeSession(WebSocketConnection connection) {
        this.lock.lock();
        try {
            if (connection != null) {
                this.webSocketConnections.remove(connection.getWebSocket());
                this.availableWebSocketConnections.remove(connection);
                this.borrowedWebSocketConnections.remove(connection);
            }
        }
        finally {
            this.callOnConnectionChanged();
            this.lock.unlock();
        }
    }

    public void close() {
        this.shuttingDown.set(true);
        this.scheduler.shutdownNow();
        try {
            this.scheduler.awaitTermination(5L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.lock.lock();
        try {
            this.webSocketConnections.values().forEach(WebSocketConnection::close);
            this.webSocketConnections.clear();
        }
        finally {
            this.callOnConnectionChanged();
            this.lock.unlock();
        }
    }

    int getTotalWebSocketConnectionCount() {
        return this.webSocketConnections.size();
    }

    int getBorrowedWebSocketConnectionCount() {
        return this.borrowedWebSocketConnections.size();
    }

    int getAvailableWebSocketConnectionCount() {
        return this.availableWebSocketConnections.size();
    }

    private void callOnConnectionChanged() {
        if (Objects.nonNull(this.totalConnectionCountConsumer)) {
            this.totalConnectionCountConsumer.accept(this.getTotalWebSocketConnectionCount());
        }
    }

    public void setOnReconnect(Callable<Void> onReconnect) {
        this.onReconnect = onReconnect;
    }
}

