/*
 * Decompiled with CFR 0.152.
 */
package de.ponton.securelistener.security;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivateKey;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.UUID;
import javax.security.auth.x500.X500Principal;

public class PontonKeystore {
    public static final String SUFFIX_CERT = ".cert";
    public static final String SUFFIX_PRIVATE_KEY = ".key";
    private final KeyStore _keystore = KeyStore.getInstance("JKS");
    private final File _keystoreFile;
    private final String _password;

    public PontonKeystore(File keystoreFile, String password) throws GeneralSecurityException {
        if (password == null || password.length() == 0) {
            throw new KeyStoreException("Password for keystore required.");
        }
        if (!keystoreFile.getParentFile().exists()) {
            throw new KeyStoreException("Keystore Folder does not exist.");
        }
        this._keystoreFile = keystoreFile;
        this._password = password;
        if (this._keystoreFile.exists()) {
            this.loadKeystore();
        } else {
            try {
                this._keystore.load(null, password.toCharArray());
                this.storeKeystore();
            }
            catch (Exception e) {
                KeyStoreException ke = new KeyStoreException(e.getMessage());
                ke.setStackTrace(e.getStackTrace());
                throw ke;
            }
        }
    }

    public int size() {
        try {
            return this._keystore.size();
        }
        catch (KeyStoreException e) {
            return 0;
        }
    }

    public ArrayList<X509Certificate> getAllCertificates() {
        try {
            ArrayList<X509Certificate> certs = new ArrayList<X509Certificate>(this._keystore.size());
            Enumeration<String> aliases = this._keystore.aliases();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                if (!this._keystore.isCertificateEntry(alias)) continue;
                certs.add((X509Certificate)this._keystore.getCertificate(alias));
            }
            return certs;
        }
        catch (KeyStoreException e) {
            return new ArrayList<X509Certificate>();
        }
    }

    public ArrayList<String> getAllCertificateAliases() {
        try {
            ArrayList<String> certs = new ArrayList<String>(this._keystore.size());
            Enumeration<String> aliases = this._keystore.aliases();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                if (!this._keystore.isCertificateEntry(alias)) continue;
                certs.add(alias);
            }
            return certs;
        }
        catch (KeyStoreException e) {
            return new ArrayList<String>();
        }
    }

    public ArrayList<String> getCACertificateAliases() {
        ArrayList<String> caIds = new ArrayList<String>();
        for (String certId : this.getAllCertificateAliases()) {
            if (certId.endsWith(SUFFIX_CERT) || certId.endsWith(SUFFIX_PRIVATE_KEY)) continue;
            caIds.add(certId);
        }
        return caIds;
    }

    public ArrayList<String> getCertificateAliases() {
        ArrayList<String> caIds = new ArrayList<String>();
        for (String certId : this.getAllCertificateAliases()) {
            if (!certId.endsWith(SUFFIX_CERT)) continue;
            caIds.add(certId);
        }
        return caIds;
    }

    public X509Certificate getCertificate(String localID) throws KeyStoreException {
        return (X509Certificate)this._keystore.getCertificate(this.buildAliasForCertificate(localID));
    }

    public X509Certificate getCertificateForPrivateKey(String localID) throws KeyStoreException {
        return (X509Certificate)this._keystore.getCertificate(this.buildAliasForPrivateKey(localID));
    }

    public X509Certificate getCA(String id) throws KeyStoreException {
        return (X509Certificate)this._keystore.getCertificate(id);
    }

    public PrivateKey getPrivateKey(String localID, char[] password) throws GeneralSecurityException {
        String id = this.buildAliasForPrivateKey(localID);
        return (PrivateKey)this._keystore.getKey(id, password);
    }

    public PrivateKey getTempPrivateKey(String localID, char[] password) throws GeneralSecurityException {
        return (PrivateKey)this._keystore.getKey(this.buildAliasForPrivateKey("TEMP_" + localID), password);
    }

    public Certificate[] getCertificateChainForPrivateKey(String localID) throws KeyStoreException {
        return this._keystore.getCertificateChain(this.buildAliasForPrivateKey(localID));
    }

    public Certificate[] getCertificateChainForTempPrivateKey(String localID) throws KeyStoreException {
        return this._keystore.getCertificateChain(this.buildAliasForPrivateKey("TEMP_" + localID));
    }

    public String addCA(X509Certificate caCert) throws KeyStoreException, CertificateException {
        return this.addCA(caCert, null);
    }

    public String addCA(X509Certificate caCert, String alias) throws KeyStoreException, CertificateException {
        try {
            this.checkCertificate(caCert);
        }
        catch (GeneralSecurityException e) {
            CertificateException ce = new CertificateException(e.getMessage());
            ce.setStackTrace(e.getStackTrace());
            throw ce;
        }
        if (alias == null) {
            alias = this.buildAliasForCA(caCert);
        }
        this._keystore.setCertificateEntry(alias, caCert);
        try {
            this.storeKeystore();
        }
        catch (KeyStoreException e) {
            this._keystore.deleteEntry(alias);
            throw e;
        }
        return alias;
    }

    public void addCertificate(String localID, X509Certificate cert, boolean checkCertificate) throws KeyStoreException, CertificateException {
        String alias = this.buildAliasForCertificate(localID);
        this.addCertificateWithAlias(alias, cert, checkCertificate);
    }

    private void addCertificateWithAlias(String alias, X509Certificate cert, boolean checkCertificate) throws CertificateException, KeyStoreException {
        try {
            if (checkCertificate) {
                this.checkCertificate(cert);
            }
        }
        catch (GeneralSecurityException e) {
            CertificateException ce = new CertificateException(e.getMessage());
            ce.setStackTrace(e.getStackTrace());
            throw ce;
        }
        this._keystore.setCertificateEntry(alias, cert);
        try {
            this.storeKeystore();
        }
        catch (KeyStoreException e) {
            this._keystore.deleteEntry(alias);
            throw e;
        }
    }

    public void checkValidity(X509Certificate certificate) throws CertificateException {
        try {
            certificate.checkValidity();
        }
        catch (CertificateExpiredException e) {
            Object msg = e.getMessage();
            msg = msg == null || ((String)msg).length() == 0 ? "." : ": " + (String)msg;
            msg = "Certificate '" + certificate.getSubjectX500Principal().getName() + "' is out of date" + (String)msg;
            CertificateException ce = new CertificateException((String)msg);
            ce.setStackTrace(e.getStackTrace());
            throw ce;
        }
        catch (CertificateNotYetValidException e) {
            Object msg = e.getMessage();
            msg = msg == null || ((String)msg).length() == 0 ? "." : ": " + (String)msg;
            msg = "Certificate '" + certificate.getSubjectX500Principal().getName() + "' is not valid" + (String)msg;
            CertificateException ce = new CertificateException((String)msg);
            ce.setStackTrace(e.getStackTrace());
            throw ce;
        }
    }

    public void addTempPrivateKey(String localID, PrivateKey privateKey, char[] password, X509Certificate certificate) throws KeyStoreException, CertificateException {
        this.addPrivateKey("TEMP_" + localID, privateKey, password, certificate);
    }

    public void deletePrivateKey(String localID) throws KeyStoreException {
        this.deleteEntry(this.buildAliasForPrivateKey(localID));
        this.deleteCertificate(localID);
        this.storeKeystore();
    }

    public void addPrivateKey(String localID, PrivateKey privateKey, char[] password, X509Certificate certificate) throws KeyStoreException, CertificateException {
        String alias = this.buildAliasForPrivateKey(localID);
        this.storePrivateKey(privateKey, password, certificate, alias);
    }

    public void addPrivateKey(String localID, PrivateKey privateKey, char[] password, X509Certificate[] certificateChain) throws KeyStoreException, CertificateException {
        String alias = this.buildAliasForPrivateKey(localID);
        this.storePrivateKey(privateKey, password, certificateChain, alias);
    }

    private void storePrivateKey(PrivateKey privateKey, char[] password, X509Certificate certificate, String alias) throws KeyStoreException, CertificateException {
        X509Certificate[] certChain = this.getCertificateChain(certificate);
        this.storePrivateKey(privateKey, password, certChain, alias);
    }

    private void storePrivateKey(PrivateKey privateKey, char[] password, X509Certificate[] certChain, String alias) throws CertificateException, KeyStoreException {
        try {
            this.checkCertificate(certChain);
        }
        catch (GeneralSecurityException e) {
            CertificateException ce = new CertificateException(e.getMessage());
            ce.setStackTrace(e.getStackTrace());
            throw ce;
        }
        this._keystore.setKeyEntry(alias, privateKey, password, certChain);
        try {
            this.storeKeystore();
        }
        catch (KeyStoreException e) {
            this._keystore.deleteEntry(alias);
            throw e;
        }
    }

    public void deleteCertificate(String localID) throws KeyStoreException {
        this.deleteEntry(this.buildAliasForCertificate(localID));
    }

    private void deleteEntry(String alias) throws KeyStoreException {
        this._keystore.deleteEntry(alias);
        this.storeKeystore();
    }

    private boolean containsAlias(String alias) throws KeyStoreException {
        return this._keystore.containsAlias(alias);
    }

    public String getCertificateAlias(X509Certificate cert) throws KeyStoreException {
        return this._keystore.getCertificateAlias(cert);
    }

    public X509Certificate[] getCertificateChain(X509Certificate cert) throws KeyStoreException, CertificateException {
        ArrayList<X509Certificate> v = new ArrayList<X509Certificate>();
        v.add(cert);
        X500Principal subject = cert.getSubjectX500Principal();
        X500Principal issuer = cert.getIssuerX500Principal();
        while (!issuer.equals(subject)) {
            X509Certificate issuerCert = this.getIssuerCertificate(cert);
            if (issuerCert == null) {
                throw new CertificateException("No certificate for CA '" + issuer.getName() + "' found.");
            }
            try {
                issuerCert.checkValidity();
            }
            catch (GeneralSecurityException e) {
                Object msg = e.getMessage();
                msg = msg == null || ((String)msg).length() == 0 ? "." : ": " + (String)msg;
                msg = "Certificate for '" + issuerCert.getIssuerX500Principal().getName() + "' is out of date" + (String)msg;
                CertificateException ce = new CertificateException((String)msg);
                ce.setStackTrace(e.getStackTrace());
                throw ce;
            }
            v.add(issuerCert);
            subject = issuerCert.getSubjectX500Principal();
            issuer = issuerCert.getIssuerX500Principal();
            cert = issuerCert;
        }
        X509Certificate[] certChain = new X509Certificate[v.size()];
        for (int i = 0; i < v.size(); ++i) {
            certChain[i] = (X509Certificate)v.get(i);
        }
        return certChain;
    }

    private X509Certificate getIssuerCertificate(X509Certificate cert) {
        X500Principal issuer = cert.getIssuerX500Principal();
        for (X509Certificate aCert : this.getAllCertificates()) {
            if (!aCert.getSubjectX500Principal().equals(issuer)) continue;
            return aCert;
        }
        return null;
    }

    public void storeKeystore() throws KeyStoreException {
        FileOutputStream out = null;
        try {
            out = new FileOutputStream(this._keystoreFile);
            this._keystore.store(out, this._password.toCharArray());
        }
        catch (Exception e) {
            Object msg = e.getMessage();
            msg = msg == null || ((String)msg).length() == 0 ? "." : ": " + (String)msg;
            msg = "Unable to store keystore" + (String)msg;
            KeyStoreException ke = new KeyStoreException((String)msg);
            ke.setStackTrace(e.getStackTrace());
            throw ke;
        }
        finally {
            if (out != null) {
                try {
                    out.close();
                }
                catch (Exception exception) {}
            }
        }
    }

    private void loadKeystore() throws KeyStoreException {
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(this._keystoreFile);
            this._keystore.load(fis, this._password.toCharArray());
        }
        catch (Exception e) {
            Object msg = e.getMessage();
            msg = msg == null || ((String)msg).length() == 0 ? "." : ": " + (String)msg;
            msg = "Unable to load keystore" + (String)msg;
            KeyStoreException ke = new KeyStoreException((String)msg);
            ke.setStackTrace(e.getStackTrace());
            throw ke;
        }
        finally {
            if (fis != null) {
                try {
                    fis.close();
                }
                catch (Exception exception) {}
            }
        }
    }

    public void checkCertificate(X509Certificate certificate, X509Certificate caCertificate) throws GeneralSecurityException {
        try {
            certificate.verify(caCertificate.getPublicKey(), "BC");
        }
        catch (InvalidKeyException e) {
            Object msg = e.getMessage();
            msg = msg == null || ((String)msg).length() == 0 ? "." : ": " + (String)msg;
            msg = "Signing CA '" + caCertificate.getSubjectX500Principal().getName() + "' is not issuer of certificate '" + certificate.getSubjectX500Principal().getName() + "'" + (String)msg;
            CertificateException ce = new CertificateException((String)msg);
            ce.setStackTrace(e.getStackTrace());
            throw ce;
        }
        catch (SignatureException e) {
            Object msg = e.getMessage();
            msg = msg == null || ((String)msg).length() == 0 ? "." : ": " + (String)msg;
            msg = "Signing CA '" + caCertificate.getSubjectX500Principal().getName() + "' is not issuer of certificate '" + certificate.getSubjectX500Principal().getName() + "'" + (String)msg;
            CertificateException ce = new CertificateException((String)msg);
            ce.setStackTrace(e.getStackTrace());
            throw ce;
        }
    }

    public void checkCertificate(X509Certificate certificate) throws GeneralSecurityException {
        this.checkCertificate(this.getCertificateChain(certificate));
    }

    public void checkCertificate(X509Certificate[] certChain) throws GeneralSecurityException {
        for (int i = 0; i < certChain.length; ++i) {
            X509Certificate certificate = certChain[i];
            this.checkValidity(certificate);
            if (i + 1 < certChain.length) {
                this.checkCertificate(certificate, certChain[i + 1]);
                continue;
            }
            this.checkCertificate(certificate, certChain[i]);
        }
    }

    private String buildAliasForPrivateKey(String localID) {
        return localID + SUFFIX_PRIVATE_KEY;
    }

    public String buildAliasForCA(X509Certificate caCertificate) {
        return caCertificate.getSubjectX500Principal().getName("CANONICAL");
    }

    private String buildAliasForCertificate(String localID) throws KeyStoreException {
        if (this.containsAlias(localID)) {
            return localID;
        }
        return localID + SUFFIX_CERT;
    }

    public String buildAliasForCertificate(X509Certificate certificate) throws CertificateEncodingException {
        return UUID.nameUUIDFromBytes(certificate.getEncoded()).toString().toLowerCase();
    }

    public KeyStore getRawKeystore() {
        return this._keystore;
    }

    public List<X509Certificate> getAllCACertificates() {
        List<X509Certificate> certificates = Collections.synchronizedList(new ArrayList());
        ArrayList<String> caCertificateAliases = this.getCACertificateAliases();
        for (String caCertificateAlias : caCertificateAliases) {
            try {
                X509Certificate caCertificate = this.getCA(caCertificateAlias);
                certificates.add(caCertificate);
            }
            catch (KeyStoreException keyStoreException) {}
        }
        return certificates;
    }
}

