/*
 * Decompiled with CFR 0.152.
 */
package de.pontonconsulting.xmlpipe.messenger.archive;

import de.pontonconsulting.common.file.FileUtils;
import de.pontonconsulting.xmlpipe.config.IFolders;
import de.pontonconsulting.xmlpipe.config.MessengerConfig;
import de.pontonconsulting.xmlpipe.messenger.archive.ArchiverException;
import de.pontonconsulting.xmlpipe.messenger.archive.IArchiver;
import de.pontonconsulting.xmlpipe.messenger.database.FileType;
import de.pontonconsulting.xmlpipe.messenger.database.MessageInfo;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.file.CopyOption;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class FileSystemArchiver
implements IArchiver {
    public static final Logger LOG = LogManager.getLogger((String)"Messenger.FileSystemArchiver");
    private final MessengerConfig messengerConfig;
    private final IFolders folders;

    public FileSystemArchiver(MessengerConfig messengerConfig, IFolders folders) {
        this.messengerConfig = messengerConfig;
        this.folders = folders;
    }

    @Override
    public List<IArchiver.ArchiveResponse> sendFilesToArchive(MessageInfo messageInfo, boolean isFailedMessage, List<IArchiver.File2Archive> files2archive) {
        try {
            this.deleteOldArchivedFiles(messageInfo, this.getArchiveFolder(messageInfo, isFailedMessage));
            File archiveFolder = this.getArchiveFolder(messageInfo, isFailedMessage);
            boolean useCompression = !isFailedMessage && this.messengerConfig.isArchiveCompression();
            return this.doArchive(files2archive, archiveFolder, useCompression);
        }
        catch (Exception e) {
            throw new ArchiverException("Can't archive message " + messageInfo.getMessageId() + " due to a " + e.getClass().getSimpleName() + ": " + e.getMessage(), e);
        }
    }

    private InputStream getFileFromArchive(String archiveReference) {
        File refFile = this.folders.convertStringToFolder(archiveReference);
        try {
            if (refFile.getParentFile().getName().endsWith(".zip")) {
                if (!refFile.getParentFile().exists()) {
                    throw new ArchiverException("Can't find Archive-Zip " + refFile.getParentFile().getAbsolutePath());
                }
                ZipFile zipFile = new ZipFile(refFile.getParentFile());
                Enumeration<? extends ZipEntry> entries = zipFile.entries();
                while (entries.hasMoreElements()) {
                    ZipEntry entry = entries.nextElement();
                    if (entry.isDirectory() || !entry.getName().equals(refFile.getName())) continue;
                    return new ZipFileInputStream(zipFile, zipFile.getInputStream(entry));
                }
                throw new ArchiverException("Can't find file in archive (" + archiveReference + ")");
            }
            if (!refFile.exists()) {
                throw new ArchiverException("Can't find Archive-File " + refFile.getAbsolutePath());
            }
            return new BufferedInputStream(Files.newInputStream(refFile.toPath(), new OpenOption[0]));
        }
        catch (IOException e) {
            throw new ArchiverException("Can't read file from archive (" + archiveReference + ")", e);
        }
    }

    @Override
    public void readFileFromArchive(String archiveReference, OutputStream outputStream) throws ArchiverException {
        try (InputStream inputStream = this.getFileFromArchive(archiveReference);){
            inputStream.transferTo(outputStream);
        }
        catch (Exception e) {
            throw new ArchiverException("Can't read file from archive (" + archiveReference + ")", e);
        }
    }

    private List<IArchiver.ArchiveResponse> doArchive(List<IArchiver.File2Archive> files2archive, File archiveFolder, boolean useCompression) throws IOException {
        if (useCompression) {
            return this.writeZipFile(files2archive, archiveFolder);
        }
        return this.writeFiles(files2archive, archiveFolder);
    }

    private List<IArchiver.ArchiveResponse> writeFiles(List<IArchiver.File2Archive> files2archive, File archiveFolder) throws IOException {
        ArrayList<IArchiver.ArchiveResponse> result = new ArrayList<IArchiver.ArchiveResponse>();
        for (IArchiver.File2Archive file2archive : files2archive) {
            File targetFile = new File(archiveFolder, file2archive.fileName());
            targetFile.getParentFile().mkdirs();
            try (InputStream is = file2archive.inputStream();){
                Files.copy(is, targetFile.toPath(), new CopyOption[0]);
            }
            result.add(new IArchiver.ArchiveResponse(this.folders.convertFolderToString(targetFile), file2archive.fileName(), file2archive.fileType()));
        }
        return result;
    }

    private List<IArchiver.ArchiveResponse> writeZipFile(List<IArchiver.File2Archive> files2archive, File archiveFolder) throws IOException {
        ArrayList<IArchiver.ArchiveResponse> result = new ArrayList<IArchiver.ArchiveResponse>();
        File zipFile = new File(archiveFolder.getParentFile(), archiveFolder.getName() + ".zip");
        zipFile.getParentFile().mkdirs();
        if (!zipFile.createNewFile()) {
            throw new IOException("ZipFile " + zipFile.getAbsolutePath() + " already exists");
        }
        try (FileChannel fileChannel = FileChannel.open(zipFile.toPath(), StandardOpenOption.WRITE, StandardOpenOption.CREATE);
             BufferedOutputStream fos = new BufferedOutputStream(Files.newOutputStream(zipFile.toPath(), new OpenOption[0]));
             BufferedOutputStream bos = new BufferedOutputStream(fos);
             ZipOutputStream zos = new ZipOutputStream(bos);
             FileLock lock = fileChannel.lock();){
            for (IArchiver.File2Archive file2archive : files2archive) {
                ZipEntry anEntry = new ZipEntry(file2archive.fileName());
                zos.putNextEntry(anEntry);
                try (InputStream is = file2archive.inputStream();){
                    is.transferTo(zos);
                }
                zos.closeEntry();
                File targetFile = new File(zipFile, file2archive.fileName());
                result.add(new IArchiver.ArchiveResponse(this.folders.convertFolderToString(targetFile), file2archive.fileName(), file2archive.fileType()));
            }
        }
        return result;
    }

    private void deleteOldArchivedFiles(MessageInfo info, File archiveFolder) {
        if (archiveFolder.exists()) {
            LOG.info("Trying to delete old message archive '{}", (Object)archiveFolder.getAbsolutePath());
            if (!FileUtils.deleteDirectory(archiveFolder)) {
                LOG.warn("Could not delete old message archive '{}", (Object)archiveFolder.getAbsolutePath());
            }
        } else {
            Path archiveZip = Path.of(archiveFolder.getAbsolutePath() + ".zip", new String[0]);
            if (archiveZip.toFile().exists()) {
                try {
                    Files.delete(archiveZip);
                }
                catch (IOException e) {
                    LOG.warn("Could not delete old message archive zip '{}", (Object)archiveZip);
                }
            }
        }
    }

    private File getArchiveFolder(MessageInfo info, boolean isFailedMessage) {
        Calendar messageDate = Calendar.getInstance();
        messageDate.setTimeInMillis(info.getTimestamp());
        String baseDir = isFailedMessage ? this.folders.getArchiveFailedFolder().getAbsolutePath() : this.folders.getArchiveFolder().getAbsolutePath();
        String direction = info.isInbound() ? "inbound" : "outbound";
        String year = String.valueOf(messageDate.get(1));
        String month = String.format("%02d", messageDate.get(2) + 1);
        String day = String.format("%02d", messageDate.get(5));
        String msgDir = String.valueOf(info.getDatabaseId());
        return Paths.get(baseDir, direction, year, month, day, msgDir).toFile();
    }

    public List<IArchiver.ArchiveResponse> createLegacyArchiveResponse(MessageInfo messageInfo) {
        ArrayList<IArchiver.ArchiveResponse> archiveResponses;
        block12: {
            archiveResponses = new ArrayList<IArchiver.ArchiveResponse>();
            if (Objects.isNull(messageInfo.getMessageFolder())) {
                return archiveResponses;
            }
            File messageFolder = this.folders.convertStringToFolder(messageInfo.getMessageFolder().toString());
            if (!messageFolder.exists()) {
                return archiveResponses;
            }
            try {
                if (messageFolder.getName().endsWith(".zip")) {
                    try (ZipFile zipFile = new ZipFile(messageFolder);){
                        Enumeration<? extends ZipEntry> zipEntries = zipFile.entries();
                        while (zipEntries.hasMoreElements()) {
                            String fileName = zipEntries.nextElement().getName();
                            File archiveFile = new File(messageFolder, fileName);
                            archiveResponses.add(new IArchiver.ArchiveResponse(this.folders.convertFolderToString(archiveFile), fileName, FileSystemArchiver.getFileType(messageInfo, archiveFile)));
                        }
                        break block12;
                    }
                }
                for (File archiveFile : Objects.requireNonNull(messageFolder.listFiles())) {
                    archiveResponses.add(new IArchiver.ArchiveResponse(this.folders.convertFolderToString(archiveFile), archiveFile.getName(), FileSystemArchiver.getFileType(messageInfo, archiveFile)));
                }
            }
            catch (IOException e) {
                throw new ArchiverException("Can't read files for message " + messageInfo.getMessageId() + " from MessageFolder " + String.valueOf(messageFolder), e);
            }
        }
        return archiveResponses;
    }

    public static FileType getFileType(MessageInfo info, File file) {
        String payloadFileName = Objects.isNull(info.getPayloadFile()) ? "" : info.getPayloadFile().getName();
        String fileName = file.getName();
        if (fileName.equals("packaging_envelope.xml") || fileName.equals("headers.txt")) {
            return FileType.PACKAGING_ENVELOPE;
        }
        if (fileName.equals("xp-backup.dat")) {
            return FileType.PAYLOAD_TECHNICAL;
        }
        if (fileName.equals(payloadFileName) || fileName.equals("payload_org.xml")) {
            return FileType.PAYLOAD;
        }
        if (fileName.equals("xp_backendmessage.xml")) {
            return FileType.BACKEND_ENVELOPE;
        }
        if (fileName.equals(payloadFileName + ".p7s") || fileName.startsWith(payloadFileName) && fileName.endsWith(".rsa") || fileName.equals(payloadFileName + ".xmlsec") || fileName.equals(payloadFileName + ".p7smime")) {
            return FileType.SIGNATURE;
        }
        if (fileName.equals(info.getSenderId() + ".cer")) {
            return FileType.CERTIFICATE;
        }
        if (fileName.equals("attachments") && file.isDirectory() || file.getParentFile().getName().equals("attachments")) {
            return FileType.ATTACHMENT;
        }
        return FileType.UNKNOWN;
    }

    @Override
    public void deleteFileFromArchive(List<String> archiveReferences2delete) throws ArchiverException {
        Map folders2delete = archiveReferences2delete.stream().map(this.folders::convertStringToFolder).map(f -> new AbstractMap.SimpleEntry<File, File>(f.getParentFile(), (File)f)).collect(Collectors.groupingBy(Map.Entry::getKey, Collectors.mapping(Map.Entry::getValue, Collectors.toList())));
        folders2delete.forEach((folder, folderFiles) -> {
            if (folder.getName().endsWith(".zip")) {
                this.deleteFromZip((File)folder, (List<File>)folderFiles);
            } else {
                this.deleteFromFolder((File)folder, (List<File>)folderFiles);
            }
        });
    }

    private void deleteFromFolder(File folder, List<File> folderFiles) throws ArchiverException {
        for (File file : folderFiles) {
            try {
                Files.delete(file.toPath());
            }
            catch (IOException e) {
                throw new ArchiverException("Can't delete ArchiveFile " + String.valueOf(file) + ": " + e.getMessage(), e);
            }
        }
        File[] filesInFolder = folder.listFiles();
        if (Objects.isNull(filesInFolder) || filesInFolder.length == 0) {
            try {
                Files.delete(folder.toPath());
            }
            catch (IOException e) {
                throw new ArchiverException("Can't delete empty ArchiveFolder " + String.valueOf(folder), e);
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void deleteFromZip(File zipFilePath, List<File> folderFiles) throws ArchiverException {
        ArrayList fileNames = new ArrayList();
        folderFiles.stream().map(File::getName).forEach(fileNames::add);
        boolean deleteWholeZip = true;
        try (ZipFile zipFile = new ZipFile(zipFilePath);){
            Enumeration<? extends ZipEntry> zipEntries = zipFile.entries();
            while (zipEntries.hasMoreElements()) {
                ZipEntry entry = zipEntries.nextElement();
                if (fileNames.remove(entry.getName())) continue;
                deleteWholeZip = false;
                break;
            }
        }
        catch (IOException e) {
            throw new ArchiverException("Can't check archiveFile's content " + String.valueOf(zipFilePath), e);
        }
        if (deleteWholeZip) {
            try {
                Files.delete(zipFilePath.toPath());
                return;
            }
            catch (IOException e) {
                throw new ArchiverException("Can't delete ZipFIle " + String.valueOf(zipFilePath), e);
            }
        }
        HashMap<String, String> env = new HashMap<String, String>();
        env.put("create", "false");
        URI zipFileUri = URI.create("jar:" + String.valueOf(zipFilePath.toURI()));
        try (FileSystem zipfs = FileSystems.newFileSystem(zipFileUri, env);){
            for (File fileInZip : folderFiles) {
                try {
                    Files.delete(zipfs.getPath(fileInZip.getName(), new String[0]));
                }
                catch (IOException e) {
                    throw new ArchiverException("Can't delete archiveFile " + fileInZip.getName() + " from ZipFile " + String.valueOf(zipFilePath), e);
                    return;
                }
            }
        }
        catch (IOException e) {
            throw new ArchiverException("Can't delete archiveFile from ZipFile " + String.valueOf(zipFilePath), e);
        }
    }

    private static class ZipFileInputStream
    extends InputStream {
        private final ZipFile zipFile;
        private final InputStream inputStream;

        public ZipFileInputStream(ZipFile zipFile, InputStream inputStream) {
            this.zipFile = zipFile;
            this.inputStream = inputStream;
        }

        @Override
        public void close() throws IOException {
            this.inputStream.close();
            this.zipFile.close();
        }

        @Override
        public int read() throws IOException {
            return this.inputStream.read();
        }

        @Override
        public byte[] readAllBytes() throws IOException {
            return this.inputStream.readAllBytes();
        }

        @Override
        public byte[] readNBytes(int len) throws IOException {
            return this.inputStream.readNBytes(len);
        }

        @Override
        public int readNBytes(byte[] b, int off, int len) throws IOException {
            return this.inputStream.readNBytes(b, off, len);
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            return this.inputStream.read(b, off, len);
        }

        @Override
        public int read(byte[] b) throws IOException {
            return this.inputStream.read(b);
        }

        @Override
        public void reset() throws IOException {
            this.inputStream.reset();
        }

        @Override
        public long skip(long n) throws IOException {
            return this.inputStream.skip(n);
        }

        @Override
        public void skipNBytes(long n) throws IOException {
            this.inputStream.skipNBytes(n);
        }

        @Override
        public long transferTo(OutputStream out) throws IOException {
            return this.inputStream.transferTo(out);
        }
    }
}

